/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/*-------------------------------------------------------------------------
 * FILE: ladata.c
 * Author: Christoph Schmidt, 03.11.97 BID R&D
 * This file contains the functions for the observer
 * functions.
 *-------------------------------------------------------------------------*/

#ifdef BEST_FIRMWARE

#include <capitype.h>

#else

#include <typedefs.h>

#endif


#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <observer.h>
#include <regconst.h>
#include <regx38.h>
#include <regx39.h>
#include <regx3a.h>
#include <regx3b.h>
#include <session.h>


static char obs_err_result_buffer[0x1000];


/* extern b_handlestype handle_array[]; *//* we need to know the handle array */

/* ------------------------------------------------------------------------
 * This function sets the observer mask
 * ------------------------------------------------------------------------ */

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestObsMaskSet(
    b_handletype handle,
    b_obsruletype obsrule,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME ("BestObsMaskSet [omset]");
  b_int8 position;
  b_int32 reg_val;
  b_int32 comb = 0;
  b_errtype err;
  b_int8 inbuf[2];              /* observerrule and value for 2x protocol */
  b_param_infotype *ParamInfo;
  /* license and handle checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* Check if obsrule is allowed and value is in range */
  B_ERRCHECK(BestParamCheck(handle, B_PARAM_OBSRULES, (b_int32) obsrule, value));

  /* Compute bit-position of obsrule. Caution: These positions correspond to
   * the HW bitpositions in E2925A. In case of E2926A the mapping to the real
   * HW bits is done in the firmware. */

  if ((err = BestParamInfoGet(handle, B_PARAM_OBSRULES, (b_int32) obsrule,
        &ParamInfo, (b_int32) B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32) B_PARAM_OBSSTATUS,
      (b_int32) obsrule, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  position = (b_int8) ParamInfo->paramrule;
  /* Remark: paramrule holds the bitposition */

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    inbuf[0] = (b_int8) position;
    inbuf[1] = (b_int8) value;
    err = BestBasicCommand(handle, CMD_OBSMASK_SET,
      inbuf, IN_OBSMASK_SET, NULL, NULL);
  }
  else
  {
    /* old protocol */

    B_ERRCHECK(BestBasicBlockRead(handle, OBS_MASK, (b_int8 *) & reg_val, 4UL));

    comb |= (b_int32) (1UL << position);
    if (value == 1UL)
      reg_val |= comb;
    else
      reg_val &= ~comb;

    return BestBasicBlockWrite(handle, OBS_MASK, (b_int8 *) & reg_val, 4UL);
  }
  B_ERRETURN(err);
}


/* ------------------------------------------------------------------------
 * This function gets the observer mask
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestObsMaskGet(
    b_handletype handle,
    b_obsruletype obsrule,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestObsMaskGet [omget]");

  b_int8 position;
  b_int32 comb = 0;
  b_int32 reg_val;
  b_errtype err;
  b_int8 in, out;               /* parameters for 2x protocol: rule and value */
  b_int16 outsize;
  b_param_infotype *ParamInfo;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  B_FCT_PARAM_NULL_POINTER_CHK(value);


  /* Check if obsrule is allowed */
  if ((err = BestParamInfoGet(handle, B_PARAM_OBSRULES,
	(b_int32)obsrule, &ParamInfo, (b_int32)B_ENUM_SEARCH)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_OBSRULES,
	(b_int32)obsrule, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  /* Compute bit-position of obsrule. Caution: These positions correspond to
   * the HW bitpositions in E2925A. In case of E2926A the mapping to the real
   * HW bits is done in the firmware. */

  position = (b_int8) ParamInfo->paramrule; /* paramrule holds the
                                             * bitposition */

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    in = position;
    outsize = OUT_OBSMASK_GET;
    B_ERRCHECK(BestBasicCommand(handle, CMD_OBSMASK_GET, &in,
        IN_OBSMASK_GET, &out, &outsize));

    assert(outsize == OUT_OBSMASK_GET);
    *value = (b_int32) out;

  }
  else
  {
    B_ERRCHECK(BestBasicBlockRead(handle, OBS_MASK, (b_int8 *) & reg_val, 4UL));
    comb |= (b_int32) (1UL << position);
    *value = (b_int32) ((reg_val & comb) >> position);
  }

  assert(*value == 1 || *value == 0);

  B_ERRETURN(B_E_OK);
}


/* ------------------------------------------------------------------------
 * This function sets the observer mask to default state
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestObsPropDefaultSet(
    b_handletype handle
)
{
  b_errtype err;
  b_int32 default_value = 0x0UL;
  /* license and handle checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    B_ERRETURN(BestBasicCommand(handle, CMD_OBSMASK_DEFSET, NULL, IN_OBSMASK_DEFSET, NULL, NULL));
  }
  else
  {
    B_ERRETURN(BestBasicBlockWrite(handle, OBS_MASK, (b_int8 *) & default_value, 4UL));
  }
}

#endif


b_errtype BestObsMaskRegSet (
    b_handletype  handle,
    b_int32       maskstat,
    b_int32       maskstat2,
    b_int32       value)
{
  B_TRY_VARS_NO_PROG;
  b_int8    set;
  b_int8    inbuf [IN_OBSMASK_REG_SET];
  b_int8ptr inptr;

  B_TRY_BEGIN {
    B_TRY_LICENSE (B_CAPABILITY_ANALYZER);
    set = (b_int8)(value ? 1 : 0);

    if (Best16BitRegisterFile (handle)) {
      inptr = BestLong2Stream (inbuf, &maskstat, 1);
      inptr = BestLong2Stream (inptr, &maskstat2, 1);
      inptr = BestByteCopy (inptr, &set, 1);
      B_TRY (BestBasicCommand (handle, CMD_OBSMASK_REG_SET,
			       inbuf, IN_OBSMASK_REG_SET,
			       NULL, NULL));
    }

    else {
      B_TRY_FAIL (B_E_NOT_E2925A);
    }
  }

  return B_TRY_RET;
}


b_errtype EXPORT BestObsMaskRegGet (
    b_handletype  handle,
    b_int32      *maskstat,
    b_int32      *maskstat2)
{
  B_TRY_VARS_NO_PROG;
  b_int8    outbuf [OUT_OBSMASK_REG_GET];
  b_int8ptr outptr;
  b_int16   outsize = OUT_OBSMASK_REG_GET;

  B_TRY_BEGIN {
    B_TRY_LICENSE (B_CAPABILITY_ANALYZER);

    if (Best16BitRegisterFile (handle)) {
      B_TRY (BestBasicCommand (handle, CMD_OBSMASK_REG_GET,
			       NULL, IN_OBSMASK_REG_GET,
			       outbuf, &outsize));
      outptr = BestStream2Long (maskstat, outbuf, 1);
      outptr = BestStream2Long (maskstat2, outptr, 1);
    }

    else {
      B_TRY_FAIL (B_E_NOT_E2925A);
    }
  }

  return B_TRY_RET;
  
}


/* ------------------------------------------------------------------------
 * This function gets the observer status
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestObsStatusGet(
    b_handletype handle,
    b_obsstatustype obsstatus,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestObsStatusGet [osget]");

  b_int8 outbuf[OUT_OBS_STATUS];
  b_int16 outsize = OUT_OBS_STATUS;
  b_int32 status_reg_val = 0UL;
  b_errtype err;
  *value = 0UL;

  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  B_FCT_PARAM_NULL_POINTER_CHK(value);

  /* CONSIDER: remove observer status info from dynamic capabilities and
     make code hardware-dependend here? */
  /* Check, wether obsstatus is a valid status at all */
  if ((err = BestParamProbe (handle, B_PARAM_OBSSTATUS,
	(b_int32)obsstatus)) != B_E_OK)
  {
    BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_OBSSTATUS,
	(b_int32)obsstatus, 0UL, 0UL, 0UL);
    B_ERRETURN(err);
  }

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    switch (obsstatus)
    {
    case B_OBS_FIRSTERR:
      assert(OUT_OBS_FIRST_READ == OUT_OBS_STATUS);
      assert(IN_OBS_FIRST_READ == IN_OBS_STATUS);
      err = BestBasicCommand(handle, CMD_OBS_FIRST_READ,
        NULL, IN_OBS_STATUS, outbuf, &outsize);
      (void) BestStream2Long(value, outbuf, 1UL);
      break;

    case B_OBS_ACCUERR:
      assert(OUT_OBS_ACCU_READ == OUT_OBS_STATUS);
      assert(IN_OBS_ACCU_READ == IN_OBS_STATUS);
      err = BestBasicCommand(handle, CMD_OBS_ACCU_READ,
        NULL, IN_OBS_STATUS, outbuf, &outsize);
      (void) BestStream2Long(value, outbuf, 1UL);
      break;

    case B_OBS_FIRSTERR2:
      assert(OUT_OBS_FIRST2_READ == OUT_OBS_STATUS);
      assert(IN_OBS_FIRST2_READ == IN_OBS_STATUS);
      err = BestBasicCommand(handle, CMD_OBS_FIRST2_READ,
        NULL, IN_OBS_STATUS, outbuf, &outsize);
      (void) BestStream2Long(value, outbuf, 1UL);
      break;

    case B_OBS_ACCUERR2:
      assert(OUT_OBS_ACCU2_READ == OUT_OBS_STATUS);
      assert(IN_OBS_ACCU2_READ == IN_OBS_STATUS);
      err = BestBasicCommand(handle, CMD_OBS_ACCU2_READ,
        NULL, IN_OBS_STATUS, outbuf, &outsize);
      (void) BestStream2Long(value, outbuf, 1UL);
      break;

    case B_OBS_OBSSTAT:
      assert(OUT_OBS_STATUS_READ == OUT_OBS_STATUS);
      assert(IN_OBS_STATUS_READ == IN_OBS_STATUS);
      err = BestBasicCommand(handle, CMD_OBS_STATUS_READ,
        NULL, IN_OBS_STATUS, outbuf, &outsize);
      (void) BestStream2Long(value, outbuf, 1UL);
      break;

    default:
      B_FCT_PARAM_ERROR(2, "Invalid Observer Register");
    }
    B_ERRETURN(err);
  }
  else
  {
    /* old protocol */

    switch (obsstatus)
    {
    case B_OBS_FIRSTERR:
      B_ERRCHECK(BestBasicBlockRead(handle, OBS_FIRST_ERROR, (b_int8 *) value, 4UL));
      break;

    case B_OBS_ACCUERR:
      B_ERRCHECK(BestBasicBlockRead(handle, OBS_ACCU_ERROR, (b_int8 *) value, 4UL));
      break;

    case B_OBS_OBSSTAT:
      B_ERRCHECK(BestBasicBlockRead(handle, OBS_STATUS, (b_int8 *) & status_reg_val, 1UL));
      *value = status_reg_val;
      break;

    default:
      B_FCT_PARAM_ERROR(2, "Invalid Observer Register");
    }                           /*lint !e788 some enums not used in switch */
  }
  B_ERRETURN(B_E_OK);
}



/* ------------------------------------------------------------------------
 * This function clears the Status of the Observer
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestObsStatusClear(
    b_handletype handle
)
{
  B_DECLARE_FUNCNAME ("BestObsStatusClear [osclear]");
  b_int32 value = 0UL;
  b_errtype err;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    err = BestBasicCommand(handle, CMD_OBS_STATUS_CLEAR, NULL, 0, NULL, NULL);
    B_ERRCHECK(err);
  }
  else
  {
    err = BestBasicBlockWrite(handle, OBS_CLEAR, (b_int8 *) & value, 1UL);
    B_ERRCHECK(err);
  }
  B_ERRETURN(err);
}


#ifndef BEST_FIRMWARE

b_errtype EXPORT BestObsRun(
    b_handletype handle
)
{
  b_errtype err;
  b_int32 value = 1UL;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle))
#if 0
  {
    /* New 2x protocol */
    B_ERRETURN(BestBasicCommand(handle, CMD_OBS_RUN, NULL, 0, NULL, NULL));
  }
  else
#endif

  {
    B_ERRETURN(BestBasicBlockWrite(handle, OBS_COMMAND, (b_int8 *) & value, 1UL));
  }

  B_ERRETURN(B_E_OK);
}



/* ------------------------------------------------------------------------
 * This function sets the observer mask
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestObsStop(
    b_handletype handle
)
{
  b_errtype err;
  b_int32 value = 0UL;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!Best16BitRegisterFile(handle))

#if 0
  {
    /* New 2x protocol */
    B_ERRETURN(BestBasicCommand(handle, CMD_OBS_STOP, NULL, 0, NULL, NULL));
  }
  else
#endif

  {
    B_ERRETURN(BestBasicBlockWrite(handle, OBS_COMMAND, (b_int8 *) & value, 1UL));
  }

  B_ERRETURN(B_E_OK);
}

#endif


/* ------------------------------------------------------------------------
 * This function returns a text and error number for a given bit position
 * ------------------------------------------------------------------------ */

#ifndef BEST_FIRMWARE

b_errtype BestObsErrInfoGet(
    b_handletype handle,
    b_int32 bitposition,
    b_charptrtype * errortext,
    b_obsruletype * obsrule
)
{
  B_DECLARE_FUNCNAME("BestObsErrInfoGet");

  static const char unimplemented[] = "Rule not implemented.";
  b_errtype err, errt;
  b_int32 i;
  b_generic_infotype *geninfo;  /* hold the generic infotype for observer
                                 * rules */
  b_param_infotype *parinfo;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  B_FCT_PARAM_NULL_POINTER_CHK(errortext);
  B_FCT_PARAM_NULL_POINTER_CHK(obsrule);

  errt = B_E_INVALID_OBS_RULE;  /* this is the default */
  *errortext = (b_charptrtype) unimplemented; /* default: is not implemented */

  /* get the generic information about observer rules */
  err = BestGenInfoGet(handle, B_PARAM_OBSRULES, &geninfo);
  B_ERRCHECK(err);

  /* run through all errors and search for the correct bitposition */
  for (i = 0; i < geninfo->num_elem; i++)
  {
    err = BestParamInfoGet(handle, B_PARAM_OBSRULES, i, &parinfo,
	(b_int32)B_INDEX_SEARCH);
    if (err)
      BestLastErrorAllParamsSet(handle, (b_int32)B_PARAM_OBSRULES, i, 0UL, 0UL, 0UL);
    B_ERRCHECK(err);

    if (parinfo->paramrule == bitposition)  /* see if the bitposition is
                                             * right */
    {
      *errortext = parinfo->param_short;  /* we found it and here is text */
      *obsrule = parinfo->proptyp.obsrule;  /* and here the obsruletype */
      errt = B_E_OK;
      break;
    }
  }

  B_ERRETURN(errt);
}

#endif

/* ------------------------------------------------------------------------
 * This function returns a textual explanation to given bit position
 * ------------------------------------------------------------------------ */

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestObsErrStringGet(
    b_handletype handle,
    b_int32 bitposition,
    b_charptrtype * errortext
)
{
  B_DECLARE_FUNCNAME("BestObsErrStringGet [oestrget]");
  b_errtype err;
  b_obsruletype dummy;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(errortext);
  B_ERRETURN(BestObsErrInfoGet(handle, bitposition, errortext, &dummy));
}

#endif


/* ------------------------------------------------------------------------
 * This function returns a string for the given error bit position
 * ------------------------------------------------------------------------ */

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestObsRuleErrTypeGet(
    b_handletype handle,
    b_int32 bitposition,
    b_obsruletype * obsrule
)
{
  B_DECLARE_FUNCNAME("BestObsRuleErrTypeGet [osruleget]");
  b_errtype err;
  b_charptrtype dummy;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(obsrule);
  B_ERRETURN(BestObsErrInfoGet(handle, bitposition, &dummy, obsrule));
}

#endif


/* ------------------------------------------------------------------------
 * This function is only kept for compatitibility reasons
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestObsRuleGet(
    b_int32 bitposition,
    b_obsruletype * obsrule
)
{
  B_DECLARE_FUNCNAME("BestObsRuleGet");

  b_errtype err;

#ifndef BEST_FIRMWARE
  b_handletype handle = INVALID_BEST_HANDLE;

#endif
  
  /* !!! ATTENTION !!!! This works only as long as the b_obsruletype in
   * typedefs.h is sorted by bit numbers !!  */

  B_FCT_PARAM_NULL_POINTER_CHK(obsrule);
  err = (bitposition > 24UL ? B_E_INVALID_OBS_RULE : B_E_OK);
  *obsrule = (b_obsruletype) bitposition;

  B_ERRETURN(err);
}

#define BIT_POSNOTFOUND	0xffffffffUL

b_errtype EXPORT BestObsBitPositionFind(b_handletype handle,
    b_int32ptr errstat,
    b_int32ptr errstat2,
    b_int32ptr bitposition)
{
  B_DECLARE_FUNCNAME ("BestObsBitPositionFind");
  
  *bitposition = 0;

  while (*bitposition < 64) {
    if (*bitposition < 32) {
      if (*errstat & (1UL << (int)*bitposition)) {
	*errstat &= ~(1UL << (int)*bitposition);
        goto found_label;
      }
    }

    else {
      if (*errstat2 & (1L << (int) (*bitposition - 32))) {
	*errstat2 &= ~(1L << (int)(*bitposition - 32));
        goto found_label;
      }
    }

    *bitposition += 1;
  }

  *bitposition = BIT_POSNOTFOUND;

found_label:
  B_ERRETURN(B_E_OK);
}


#ifndef BEST_FIRMWARE

b_errtype EXPORT BestObsErrResultGet(b_handletype handle,
    b_int32 errstat,
    b_int32 errstat2,
    b_ccharptrtype * errortext)
{
  B_TRY_VARS_NO_PROG;
  b_int32 bitposition = 0UL;
  b_obsruletype rule;
  B_TRY_BEGIN
  {
    obs_err_result_buffer[0] = '\0';

    do
    {
      B_TRY(BestObsBitPositionFind(handle, &errstat, &errstat2, &bitposition));
      B_TRY(BestObsErrInfoGet(handle, bitposition,
          (b_charptrtype *) errortext, &rule));
      strcat(obs_err_result_buffer, *errortext);
      strcat(obs_err_result_buffer, "\n");
    } while (bitposition != BIT_POSNOTFOUND);
  }

  B_TRY_CATCH
  {
    if (bitposition == BIT_POSNOTFOUND)
      B_TRY_RET = B_E_OK;

    if (!strlen(obs_err_result_buffer))
      strcpy(obs_err_result_buffer, "no errors\n");

    *errortext = obs_err_result_buffer;
  }

  strcat (obs_err_result_buffer, "\n");
  B_ERRETURN(B_TRY_RET);
}

#endif

